import os
import json
from redteamcore import Resource

class SELinuxError(Exception):
    def __init__(self, mode):
        super(SELinuxError, self).__init__("SELinux mode {0} is not permissive or enforcing".format(mode))

class CuratedExploit(Resource):
    def __init__(self, location, source, eid):
        super(CuratedExploit, self).__init__(location)
        self.source = source
        self.eid = eid
        
    @classmethod
    def from_eid(cls, curated_dir, source, eid):
        location = os.path.join(curated_dir, source, eid + '.json')
        return cls(location, source, eid)

    def reconcile(self, exploit_source):
        data = dict()

        data['exploit'] = exploit_source['exploit']
        data['cves'] = exploit_source['cves']
        data['source'] = exploit_source['source']
        data['id'] = exploit_source['id']
        
        self.data = data

    def cpes(self):
        self.conditional_read()
        if 'cpes' in self.data.keys():
            return self.data['cpes'].keys()
        return []

    def add_cpe(self, cpe):
        self.conditional_read()
        if 'cpes' not in self.data.keys():
            self.data['cpes'] = dict()

        if cpe not in self.cpes():
            self.data['cpes'][cpe] = dict()

    def delete_cpe(self, cpe):
        self.conditional_read()
        self.data['cpes'].pop(cpe)

    def score(self, cpe, kind, value):
        self.conditional_read()
        self.add_cpe(cpe)

        if 'scores' not in self.data['cpes'][cpe].keys():
            self.data['cpes'][cpe]['scores'] = dict()

        self.data['cpes'][cpe]['scores'][kind] = value

    def delete_score(self, cpe, kind):
        self.conditional_read()
        self.data['cpes'][cpe]['scores'].pop(kind)

    def get_score(self, cpe, kind):
        self.conditional_read()
        return self.data['cpes'][cpe]['scores'][kind]

    def set_command(self, cpe, command):
        self.conditional_read()
        self.add_cpe(cpe)
        self.data['cpes'][cpe]['staging'] = command

    def get_command(self, cpe):
        self.conditional_read()
        return self.data['cpes'][cpe]['staging']

    def delete_command(self, cpe):
        self.conditional_read()
        self.data['cpes'][cpe]['staging'] = None

    def set_packages(self, cpe, packages):
        self.conditional_read()
        self.add_cpe(cpe)
        self.data['cpes'][cpe]['packages'] = packages

    def add_packages(self, cpe, packages):
        if isinstance(packages, str):
            packages = [packages]
        self.conditional_read()
        self.data['cpes'][cpe]['packages'] = list(set(self.data['cpes'][cpe]['packages'] + packages))

    def get_packages(self, cpe):
        self.conditional_read()
        return self.data['cpes'][cpe]['packages']

    def delete_packages(self, cpe, packages=None):
        self.conditional_read()
        if isinstance(packages, str):
            packages = [packages]

        if packages is None:
            self.data['cpes'][cpe]['packages'] = None
        else:
            packages = [package for package in self.data['cpes'][cpe]['packages'] if package not in packages]
            self.data['cpes'][cpe]['packages'] = packages

    def set_services(self, cpe, services):
        self.conditional_read()
        self.add_cpe(cpe)
        self.data['cpes'][cpe]['services'] = services

    def add_services(self, cpe, services):
        if isinstance(services, str):
            services = [services]
        self.conditional_read()
        self.data['cpes'][cpe]['services'] = list(set(self.data['cpes'][cpe]['services'] + services))

    def get_services(self, cpe):
        self.conditional_read()
        return self.data['cpes'][cpe]['services']

    def delete_services(self, cpe, services=None):
        self.conditional_read()
        if isinstance(services, str):
            services = [services]

        if services is None:
            self.data['cpes'][cpe]['services'] = None
        else:
            services = [service for service in self.data['cpes'][cpe]['services'] if service not in services]
            self.data['cpes'][cpe]['services'] = services

    def set_selinux(self, cpe, selinux):
        if selinux not in ['permissive', 'enforcing']:
            raise SELinuxError(selinux)

        self.conditional_read()
        self.add_cpe(cpe)
        self.data['cpes'][cpe]['selinux'] = selinux

    def get_selinux(self, cpe):
        self.conditional_read()
        return self.data['cpes'][cpe]['selinux']

    def delete_selinux(self, cpe):
        self.conditional_read()
        self.data['cpes'][cpe]['selinux'] = None

    def set_filename(self, cpe, filename):
        self.conditional_read()
        self.add_cpe(cpe)
        self.data['cpes'][cpe]['filename'] = filename

    def get_filename(self, cpe):
        self.conditional_read()

        try:
            filename = self.data['cpes'][cpe]['filename']
        except KeyError:
            filename = self.source + "-" + self.eid + '.txt'
        return filename

    def delete_filename(self, cpe):
        self.conditional_read()
        self.data['cpes'][cpe]['filename'] = None

    def set_exploit(self, base64_exploit):
        self.conditional_read()
        self.data['exploit'] = base64_exploit

    def get_exploit(self):
        self.conditional_read()
        return self.data['exploit']

    def cves(self):
        self.conditional_read()
        return self.data['cves']

    def conditional_read(self):
        if not self.data:
            self.data = self.read()

    def write(self):
        self.conditional_read()
        with open(self.location, 'w') as curation_file_obj:
            json.dump(self.data, curation_file_obj, indent=4)
