/*
 * Decompiled with CFR 0.152.
 */
package com.paterva.maltego.certificates.impl;

import com.paterva.maltego.certificates.CertificateDefinition;
import com.paterva.maltego.certificates.CertificateRepository;
import com.paterva.maltego.certificates.CertificateUtils;
import com.paterva.maltego.util.FileUtilities;
import com.paterva.maltego.util.StringUtilities;
import com.paterva.maltego.util.Version;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.security.GeneralSecurityException;
import java.security.InvalidKeyException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.SignatureException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.apache.commons.codec.digest.DigestUtils;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileUtil;
import org.openide.util.Exceptions;

public class FileSystemCertificateRepository
extends CertificateRepository {
    private static final Logger LOG = Logger.getLogger(FileSystemCertificateRepository.class.getName());
    private static final String CERTIFICATES_FOLDER = "Maltego/Certificates";
    private static final String EXTENSION = "certificate";
    private static final String CTAS_CERT = "-----BEGIN CERTIFICATE-----\nMIIDRzCCAi+gAwIBAgIJAN+tgZwBm5v0MA0GCSqGSIb3DQEBBQUAMDoxFDASBgNV\nBAMMC01BTFRFR09fU1JWMSIwIAYJKoZIhvcNAQkBFhNtYWx0ZWdvQHBhdGVydmEu\nY29tMB4XDTEyMTEzMDEwMzAyNVoXDTIyMTEyODEwMzAyNVowOjEUMBIGA1UEAwwL\nTUFMVEVHT19TUlYxIjAgBgkqhkiG9w0BCQEWE21hbHRlZ29AcGF0ZXJ2YS5jb20w\nggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCvf0k4g8vxnd7aMYocysUG\nHo0qsm83sPTQi5fC8jsvndgBdzWC7ZpPdwwWSp3tm2Gbl7LZZswfvkFRP6vLm7o9\nBbMnVFOBVsjHVmQ/VXeM6ZtkgbaC/nh7YqalY+MsSm49wh7o1qI7rl2ByiipLdP7\ncva2iaW1ViLZfBvZODa4jAyy3KbrMqDxHDgQvVlkB8v/8IzkexBDniz2wcpq4b2N\nVzyrPxO/RfY9icYIGs2QaL3fOLmsKodTM9u5ezR525F9RmX4xV+mVd6qQRLnU9eq\nwNhQLRg3ENFRDx2Er+iHowMDAzLdjuQhNz6ikYXKLfFXxaZAhs+MpeXKgbK/a/5d\nAgMBAAGjUDBOMB0GA1UdDgQWBBR44uOKtz88tYgPkzuNMpQMGkdJdDAfBgNVHSME\nGDAWgBR44uOKtz88tYgPkzuNMpQMGkdJdDAMBgNVHRMEBTADAQH/MA0GCSqGSIb3\nDQEBBQUAA4IBAQBMM/8ZSyPz9dOZDgCDAnbufZRPqdhZ7INZE/1v7+PXJl6UdJVQ\n7XK9oxe/v6rXuqxbS5dTBqgUIDEf16tEpZctF5AyqiVJIfrbg0wQyR910+3fLwn0\nv2hh76f+GGBQTCzolaqHHpa4n6PrWgi8HP2loWsvi0Yk6v3QNugZ32S5B2gVZJjb\nG4VtnctGvL4RVkTK+K99ap/8nPIdzCp85HgEPpF+MOptxKpmWZIe7wwAUnXyFF6y\nTOM9dwD9RpEr7dl38JQZQ+E6gb6BQFMjOHPpFL6HtSQpAKgsKmnjpT9+AFhg2+/c\nadnkYJ30iKDB1iCoEt8mYIvu4p/rcb+T3+Xq\n-----END CERTIFICATE-----";
    private static final String PATERVA_CA_CERT = "-----BEGIN CERTIFICATE-----\nMIIEzTCCA7WgAwIBAgIJALFS50/cG0TOMA0GCSqGSIb3DQEBBQUAMIGfMQswCQYD\nVQQGEwJaQTEQMA4GA1UECBMHR2F1dGVuZzEOMAwGA1UEBxMFSXJlbmUxGDAWBgNV\nBAoTD1BhdGVydmEgUHR5IEx0ZDEgMB4GA1UECxMXQ3J5cHRvIGNvbnRyb2wgZGl2\naXNpb24xEzARBgNVBAMTClBhdGVydmEgQ0ExHTAbBgkqhkiG9w0BCQEWDmNhQHBh\ndGVydmEuY29tMB4XDTEwMDUxOTEwNTIxOFoXDTIwMDUxNjEwNTIxOFowgZ8xCzAJ\nBgNVBAYTAlpBMRAwDgYDVQQIEwdHYXV0ZW5nMQ4wDAYDVQQHEwVJcmVuZTEYMBYG\nA1UEChMPUGF0ZXJ2YSBQdHkgTHRkMSAwHgYDVQQLExdDcnlwdG8gY29udHJvbCBk\naXZpc2lvbjETMBEGA1UEAxMKUGF0ZXJ2YSBDQTEdMBsGCSqGSIb3DQEJARYOY2FA\ncGF0ZXJ2YS5jb20wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDJbNAE\nFEmMTvtOwRMxIcI/3RJaJhdgozm2xyuwj4AXpNnTjMElfr7fyzNT/7tk+0Ror7PK\n9bRkD6vyMrk21uJtTqy87/O7VMizF7MRknCbZIvIcb+cTqCgz93hP1CzLqEj3b87\n3oz7mDVzeS5+kFlFz1YO2Fij80buSRKk0xjSqQGL8bGIV/OXdjTb5lZ2xf9YEGRB\n0AvxahYjizXwla9NJV0OE6cTIxrRl4QT9FLbniAsl3HQTZ+eJO11MJe5fPj7S7Qp\nj17ii64kJK9ACDMDtpAjK/HQ90yD3SRERiLNEmNzY2JkUNO/YfkTI9vfOFVMtMhK\njAwx1TLe28wtq49FAgMBAAGjggEIMIIBBDAdBgNVHQ4EFgQUwJpqRnyG6bR/6dVq\nB+n3N7Y3UWAwgdQGA1UdIwSBzDCByYAUwJpqRnyG6bR/6dVqB+n3N7Y3UWChgaWk\ngaIwgZ8xCzAJBgNVBAYTAlpBMRAwDgYDVQQIEwdHYXV0ZW5nMQ4wDAYDVQQHEwVJ\ncmVuZTEYMBYGA1UEChMPUGF0ZXJ2YSBQdHkgTHRkMSAwHgYDVQQLExdDcnlwdG8g\nY29udHJvbCBkaXZpc2lvbjETMBEGA1UEAxMKUGF0ZXJ2YSBDQTEdMBsGCSqGSIb3\nDQEJARYOY2FAcGF0ZXJ2YS5jb22CCQCxUudP3BtEzjAMBgNVHRMEBTADAQH/MA0G\nCSqGSIb3DQEBBQUAA4IBAQA/BD1NHz5PGENv2iF7wydRktY+qvTNNJoAI8zRW7AY\nsu+ATmAJ1Lljmgy6hjIUl/aHZrcO/6/TNQVSilxqRnMBGBH8eMUJYuPJpiZxU2Ss\n2co4Pv5LI3PVBuvlJwEqmMYy+iNIeD0scto304nYEN1kPDivt9vrkoQD+0bVMPCo\n82Smzu4h6lkDiAdof0svAK9+1nWvzoXimsVdXyH/vkHPWXAt2qw1sG9Qg2CxQ6t6\ntxuph9Yp9li/fdsTyIgMA/gV3iX4Mph4hFcD2VxKf2t/WVDLsEdS7J6l5v8zoFTb\n/4QOdtL5Vt7i1c4jZUBdyISzI5oNhA2EGty3RTyhFwCs\n-----END CERTIFICATE-----";
    private static final String CTAS_CERT_2019 = "-----BEGIN CERTIFICATE-----\nMIIDyTCCArECCQCmut0hmXjH1DANBgkqhkiG9w0BAQsFADCBojELMAkGA1UEBhMC\nWkExEDAOBgNVBAgMB0dhdXRlbmcxETAPBgNVBAcMCEJvc2Noa29wMRgwFgYDVQQK\nDA9QYXRlcnZhIFB0eSBMdGQxIDAeBgNVBAsMF0NyeXB0byBjb250cm9sIGRpdmlz\naW9uMRMwEQYDVQQDDApQYXRlcnZhIENBMR0wGwYJKoZIhvcNAQkBFg5jYUBwYXRl\ncnZhLmNvbTAeFw0xOTEwMDcxMjE2NDNaFw0yOTEwMDQxMjE2NDNaMIGpMQswCQYD\nVQQGEwJaQTEQMA4GA1UECAwHR2F1dGVuZzERMA8GA1UEBwwIQm9zY2hrb3AxGTAX\nBgNVBAoMECBQYXRlcnZhIFB0eSBMdGQxIDAeBgNVBAsMF0NyeXB0byBjb250cm9s\nIGRpdmlzaW9uMRQwEgYDVQQDDAtNQUxURUdPX1NSVjEiMCAGCSqGSIb3DQEJARYT\nbWFsdGVnb0BwYXRlcnZhLmNvbTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoC\nggEBAMMSuQPcdNPArPOHuI7e0IivP69BPasess04ENe+mNSboGxaLI3mifnGwiO4\nIbaGHUiRajLMvAY/h3YLlqtTzuY4KKoEtpio/lB7TDExPaP30Y36StDmsSyT/A91\n9gMZQFo0NMftJd5dIS6JrrjDQHqhKUDLUoffH4LDCQtCU58dLMitEzrFz6PfAWuS\nZoBv3fwZOcDEXEefO7rIFqHeMkIHffYu0gIdGLXC4I6CYCsU/4dHt2u3lSCjzZQX\nyr4wSnfGSHEjKjz6R4LtP5xjEvq9fRUToebbALCmT/oGIlXvrhE8p/zMLbHl5Hgl\nz7TGim+KfXIeWyMAsutww3nCvukCAwEAATANBgkqhkiG9w0BAQsFAAOCAQEAVFSP\np6mnuGnYGk5nlkQl1HxG2koCyS5pPzW6S/p1D1H6IKz464aXtVUdNIKGzAKjhtd+\nRTVIY2/PtWitp6fdsEZuVZj7zqZQW2F/Ld48AO5eiGmskCxu//P9gK3qITBMncsk\n9KiZnISpNNbDLOGbuNrdC6rSLGmhZwN/d1vzYPUIyobsWEo4XLohI439RZuJt4Iu\nD+SxORvF+VGM2sZxjT1YmwkkgbKYIiZpKmkiD888BJvQWGrOxeAyx/xPGgT4hlRt\n9PSgP/ZcKM3eDJldQjEIX2Cf9HbUgdj1A2R0hDEeM5c9QeR1PpsKqWuH+YTXXGjj\nu7osyZs7Sxt7lsjWfw==-----END CERTIFICATE-----";
    private static final String PATERVA_CA_CERT_2019 = "-----BEGIN CERTIFICATE-----\nMIIEHDCCAwSgAwIBAgIJAOR0c8hqretkMA0GCSqGSIb3DQEBCwUAMIGiMQswCQYD\nVQQGEwJaQTEQMA4GA1UECAwHR2F1dGVuZzERMA8GA1UEBwwIQm9zY2hrb3AxGDAW\nBgNVBAoMD1BhdGVydmEgUHR5IEx0ZDEgMB4GA1UECwwXQ3J5cHRvIGNvbnRyb2wg\nZGl2aXNpb24xEzARBgNVBAMMClBhdGVydmEgQ0ExHTAbBgkqhkiG9w0BCQEWDmNh\nQHBhdGVydmEuY29tMB4XDTE5MTAwNzExNDEzM1oXDTI5MTAwNDExNDEzM1owgaIx\nCzAJBgNVBAYTAlpBMRAwDgYDVQQIDAdHYXV0ZW5nMREwDwYDVQQHDAhCb3NjaGtv\ncDEYMBYGA1UECgwPUGF0ZXJ2YSBQdHkgTHRkMSAwHgYDVQQLDBdDcnlwdG8gY29u\ndHJvbCBkaXZpc2lvbjETMBEGA1UEAwwKUGF0ZXJ2YSBDQTEdMBsGCSqGSIb3DQEJ\nARYOY2FAcGF0ZXJ2YS5jb20wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB\nAQDRbOdCg1cE1wBqJKeVcDeL7v3qbB7QNc2NwyOvRAiJosU0wFfb9457tnAoYJvt\nEWCuCfqSMP0FAEY0imUdT7SG2Jbcb2yFXpsunkNjl7Qd1OCNNvsAH2EtnmPVIyG8\nU4vbc8/0rSZQcObVm8qWsoRqfCcWaN9YCZdrVBVdIvB1dujHPxicPgS1x7tFYIgx\nihRf7ruzxuGfTIwy/gykPHr/xhZdaXV7abONueGLHHPU2qijI7rVFx44cNb3YZzV\naCSy8vy3Xe88OTaHYKCUbHBb437zt0Tp6+6mU0eJYdfyefJ/pRoAt0yYK43EfaYu\noKWBkZA7yEnUo4Cvl/PxHBDPAgMBAAGjUzBRMB0GA1UdDgQWBBSwmjWB/tHzbgkC\n+qLwuz5vE8HR3TAfBgNVHSMEGDAWgBSwmjWB/tHzbgkC+qLwuz5vE8HR3TAPBgNV\nHRMBAf8EBTADAQH/MA0GCSqGSIb3DQEBCwUAA4IBAQAGxLwR7GlzAA1JMjmDIJFv\nj+9m2TCcknUcVQ0u6ju1uhO6Db+W/J6Fe535md2Ps6Zvyh3raGNbSEYPUWnelY7v\nX2Sr1KEWFFAh8HwIbWt8Zj84r/lJzAmAtTJEA2KGmowfDcU6ZJ77O0nEHXGLOqoH\nDDO0dRXbnkaY/7SEdcS7H3UsbgWSKV9vDeoWtxYPprE0yD4TykYNX4MUr9Qwzw6S\nBRFOeSuGXKGqJJGtnkPpRVeB7fbQ472O7b2aased9zffrpr6rDSmMgCdT7bjWsuh\nS7PeYH7YTfuV9B3b/kaioWe72sZE5x05zP6flBCOxY/7bbK2T/ZjgYHXvT3BX6+r-----END CERTIFICATE-----";
    private Map<X509Certificate, CertificateDefinition> _certMap;
    private static final String ATTR_X509CERTIFICATE = "X509Certificate";
    private static final String ATTR_HOSTNAMES = "hostnames";
    private static final String ATTR_TRUSTED = "trusted";
    private final X509Certificate _ctasCert = this.toCertificate("-----BEGIN CERTIFICATE-----\nMIIDRzCCAi+gAwIBAgIJAN+tgZwBm5v0MA0GCSqGSIb3DQEBBQUAMDoxFDASBgNV\nBAMMC01BTFRFR09fU1JWMSIwIAYJKoZIhvcNAQkBFhNtYWx0ZWdvQHBhdGVydmEu\nY29tMB4XDTEyMTEzMDEwMzAyNVoXDTIyMTEyODEwMzAyNVowOjEUMBIGA1UEAwwL\nTUFMVEVHT19TUlYxIjAgBgkqhkiG9w0BCQEWE21hbHRlZ29AcGF0ZXJ2YS5jb20w\nggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCvf0k4g8vxnd7aMYocysUG\nHo0qsm83sPTQi5fC8jsvndgBdzWC7ZpPdwwWSp3tm2Gbl7LZZswfvkFRP6vLm7o9\nBbMnVFOBVsjHVmQ/VXeM6ZtkgbaC/nh7YqalY+MsSm49wh7o1qI7rl2ByiipLdP7\ncva2iaW1ViLZfBvZODa4jAyy3KbrMqDxHDgQvVlkB8v/8IzkexBDniz2wcpq4b2N\nVzyrPxO/RfY9icYIGs2QaL3fOLmsKodTM9u5ezR525F9RmX4xV+mVd6qQRLnU9eq\nwNhQLRg3ENFRDx2Er+iHowMDAzLdjuQhNz6ikYXKLfFXxaZAhs+MpeXKgbK/a/5d\nAgMBAAGjUDBOMB0GA1UdDgQWBBR44uOKtz88tYgPkzuNMpQMGkdJdDAfBgNVHSME\nGDAWgBR44uOKtz88tYgPkzuNMpQMGkdJdDAMBgNVHRMEBTADAQH/MA0GCSqGSIb3\nDQEBBQUAA4IBAQBMM/8ZSyPz9dOZDgCDAnbufZRPqdhZ7INZE/1v7+PXJl6UdJVQ\n7XK9oxe/v6rXuqxbS5dTBqgUIDEf16tEpZctF5AyqiVJIfrbg0wQyR910+3fLwn0\nv2hh76f+GGBQTCzolaqHHpa4n6PrWgi8HP2loWsvi0Yk6v3QNugZ32S5B2gVZJjb\nG4VtnctGvL4RVkTK+K99ap/8nPIdzCp85HgEPpF+MOptxKpmWZIe7wwAUnXyFF6y\nTOM9dwD9RpEr7dl38JQZQ+E6gb6BQFMjOHPpFL6HtSQpAKgsKmnjpT9+AFhg2+/c\nadnkYJ30iKDB1iCoEt8mYIvu4p/rcb+T3+Xq\n-----END CERTIFICATE-----");
    private final X509Certificate _patervaCaCert = this.toCertificate("-----BEGIN CERTIFICATE-----\nMIIEzTCCA7WgAwIBAgIJALFS50/cG0TOMA0GCSqGSIb3DQEBBQUAMIGfMQswCQYD\nVQQGEwJaQTEQMA4GA1UECBMHR2F1dGVuZzEOMAwGA1UEBxMFSXJlbmUxGDAWBgNV\nBAoTD1BhdGVydmEgUHR5IEx0ZDEgMB4GA1UECxMXQ3J5cHRvIGNvbnRyb2wgZGl2\naXNpb24xEzARBgNVBAMTClBhdGVydmEgQ0ExHTAbBgkqhkiG9w0BCQEWDmNhQHBh\ndGVydmEuY29tMB4XDTEwMDUxOTEwNTIxOFoXDTIwMDUxNjEwNTIxOFowgZ8xCzAJ\nBgNVBAYTAlpBMRAwDgYDVQQIEwdHYXV0ZW5nMQ4wDAYDVQQHEwVJcmVuZTEYMBYG\nA1UEChMPUGF0ZXJ2YSBQdHkgTHRkMSAwHgYDVQQLExdDcnlwdG8gY29udHJvbCBk\naXZpc2lvbjETMBEGA1UEAxMKUGF0ZXJ2YSBDQTEdMBsGCSqGSIb3DQEJARYOY2FA\ncGF0ZXJ2YS5jb20wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDJbNAE\nFEmMTvtOwRMxIcI/3RJaJhdgozm2xyuwj4AXpNnTjMElfr7fyzNT/7tk+0Ror7PK\n9bRkD6vyMrk21uJtTqy87/O7VMizF7MRknCbZIvIcb+cTqCgz93hP1CzLqEj3b87\n3oz7mDVzeS5+kFlFz1YO2Fij80buSRKk0xjSqQGL8bGIV/OXdjTb5lZ2xf9YEGRB\n0AvxahYjizXwla9NJV0OE6cTIxrRl4QT9FLbniAsl3HQTZ+eJO11MJe5fPj7S7Qp\nj17ii64kJK9ACDMDtpAjK/HQ90yD3SRERiLNEmNzY2JkUNO/YfkTI9vfOFVMtMhK\njAwx1TLe28wtq49FAgMBAAGjggEIMIIBBDAdBgNVHQ4EFgQUwJpqRnyG6bR/6dVq\nB+n3N7Y3UWAwgdQGA1UdIwSBzDCByYAUwJpqRnyG6bR/6dVqB+n3N7Y3UWChgaWk\ngaIwgZ8xCzAJBgNVBAYTAlpBMRAwDgYDVQQIEwdHYXV0ZW5nMQ4wDAYDVQQHEwVJ\ncmVuZTEYMBYGA1UEChMPUGF0ZXJ2YSBQdHkgTHRkMSAwHgYDVQQLExdDcnlwdG8g\nY29udHJvbCBkaXZpc2lvbjETMBEGA1UEAxMKUGF0ZXJ2YSBDQTEdMBsGCSqGSIb3\nDQEJARYOY2FAcGF0ZXJ2YS5jb22CCQCxUudP3BtEzjAMBgNVHRMEBTADAQH/MA0G\nCSqGSIb3DQEBBQUAA4IBAQA/BD1NHz5PGENv2iF7wydRktY+qvTNNJoAI8zRW7AY\nsu+ATmAJ1Lljmgy6hjIUl/aHZrcO/6/TNQVSilxqRnMBGBH8eMUJYuPJpiZxU2Ss\n2co4Pv5LI3PVBuvlJwEqmMYy+iNIeD0scto304nYEN1kPDivt9vrkoQD+0bVMPCo\n82Smzu4h6lkDiAdof0svAK9+1nWvzoXimsVdXyH/vkHPWXAt2qw1sG9Qg2CxQ6t6\ntxuph9Yp9li/fdsTyIgMA/gV3iX4Mph4hFcD2VxKf2t/WVDLsEdS7J6l5v8zoFTb\n/4QOdtL5Vt7i1c4jZUBdyISzI5oNhA2EGty3RTyhFwCs\n-----END CERTIFICATE-----");
    private final X509Certificate _ctasCert2019 = this.toCertificate("-----BEGIN CERTIFICATE-----\nMIIDyTCCArECCQCmut0hmXjH1DANBgkqhkiG9w0BAQsFADCBojELMAkGA1UEBhMC\nWkExEDAOBgNVBAgMB0dhdXRlbmcxETAPBgNVBAcMCEJvc2Noa29wMRgwFgYDVQQK\nDA9QYXRlcnZhIFB0eSBMdGQxIDAeBgNVBAsMF0NyeXB0byBjb250cm9sIGRpdmlz\naW9uMRMwEQYDVQQDDApQYXRlcnZhIENBMR0wGwYJKoZIhvcNAQkBFg5jYUBwYXRl\ncnZhLmNvbTAeFw0xOTEwMDcxMjE2NDNaFw0yOTEwMDQxMjE2NDNaMIGpMQswCQYD\nVQQGEwJaQTEQMA4GA1UECAwHR2F1dGVuZzERMA8GA1UEBwwIQm9zY2hrb3AxGTAX\nBgNVBAoMECBQYXRlcnZhIFB0eSBMdGQxIDAeBgNVBAsMF0NyeXB0byBjb250cm9s\nIGRpdmlzaW9uMRQwEgYDVQQDDAtNQUxURUdPX1NSVjEiMCAGCSqGSIb3DQEJARYT\nbWFsdGVnb0BwYXRlcnZhLmNvbTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoC\nggEBAMMSuQPcdNPArPOHuI7e0IivP69BPasess04ENe+mNSboGxaLI3mifnGwiO4\nIbaGHUiRajLMvAY/h3YLlqtTzuY4KKoEtpio/lB7TDExPaP30Y36StDmsSyT/A91\n9gMZQFo0NMftJd5dIS6JrrjDQHqhKUDLUoffH4LDCQtCU58dLMitEzrFz6PfAWuS\nZoBv3fwZOcDEXEefO7rIFqHeMkIHffYu0gIdGLXC4I6CYCsU/4dHt2u3lSCjzZQX\nyr4wSnfGSHEjKjz6R4LtP5xjEvq9fRUToebbALCmT/oGIlXvrhE8p/zMLbHl5Hgl\nz7TGim+KfXIeWyMAsutww3nCvukCAwEAATANBgkqhkiG9w0BAQsFAAOCAQEAVFSP\np6mnuGnYGk5nlkQl1HxG2koCyS5pPzW6S/p1D1H6IKz464aXtVUdNIKGzAKjhtd+\nRTVIY2/PtWitp6fdsEZuVZj7zqZQW2F/Ld48AO5eiGmskCxu//P9gK3qITBMncsk\n9KiZnISpNNbDLOGbuNrdC6rSLGmhZwN/d1vzYPUIyobsWEo4XLohI439RZuJt4Iu\nD+SxORvF+VGM2sZxjT1YmwkkgbKYIiZpKmkiD888BJvQWGrOxeAyx/xPGgT4hlRt\n9PSgP/ZcKM3eDJldQjEIX2Cf9HbUgdj1A2R0hDEeM5c9QeR1PpsKqWuH+YTXXGjj\nu7osyZs7Sxt7lsjWfw==-----END CERTIFICATE-----");
    private final X509Certificate _patervaCaCert2019 = this.toCertificate("-----BEGIN CERTIFICATE-----\nMIIEHDCCAwSgAwIBAgIJAOR0c8hqretkMA0GCSqGSIb3DQEBCwUAMIGiMQswCQYD\nVQQGEwJaQTEQMA4GA1UECAwHR2F1dGVuZzERMA8GA1UEBwwIQm9zY2hrb3AxGDAW\nBgNVBAoMD1BhdGVydmEgUHR5IEx0ZDEgMB4GA1UECwwXQ3J5cHRvIGNvbnRyb2wg\nZGl2aXNpb24xEzARBgNVBAMMClBhdGVydmEgQ0ExHTAbBgkqhkiG9w0BCQEWDmNh\nQHBhdGVydmEuY29tMB4XDTE5MTAwNzExNDEzM1oXDTI5MTAwNDExNDEzM1owgaIx\nCzAJBgNVBAYTAlpBMRAwDgYDVQQIDAdHYXV0ZW5nMREwDwYDVQQHDAhCb3NjaGtv\ncDEYMBYGA1UECgwPUGF0ZXJ2YSBQdHkgTHRkMSAwHgYDVQQLDBdDcnlwdG8gY29u\ndHJvbCBkaXZpc2lvbjETMBEGA1UEAwwKUGF0ZXJ2YSBDQTEdMBsGCSqGSIb3DQEJ\nARYOY2FAcGF0ZXJ2YS5jb20wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB\nAQDRbOdCg1cE1wBqJKeVcDeL7v3qbB7QNc2NwyOvRAiJosU0wFfb9457tnAoYJvt\nEWCuCfqSMP0FAEY0imUdT7SG2Jbcb2yFXpsunkNjl7Qd1OCNNvsAH2EtnmPVIyG8\nU4vbc8/0rSZQcObVm8qWsoRqfCcWaN9YCZdrVBVdIvB1dujHPxicPgS1x7tFYIgx\nihRf7ruzxuGfTIwy/gykPHr/xhZdaXV7abONueGLHHPU2qijI7rVFx44cNb3YZzV\naCSy8vy3Xe88OTaHYKCUbHBb437zt0Tp6+6mU0eJYdfyefJ/pRoAt0yYK43EfaYu\noKWBkZA7yEnUo4Cvl/PxHBDPAgMBAAGjUzBRMB0GA1UdDgQWBBSwmjWB/tHzbgkC\n+qLwuz5vE8HR3TAfBgNVHSMEGDAWgBSwmjWB/tHzbgkC+qLwuz5vE8HR3TAPBgNV\nHRMBAf8EBTADAQH/MA0GCSqGSIb3DQEBCwUAA4IBAQAGxLwR7GlzAA1JMjmDIJFv\nj+9m2TCcknUcVQ0u6ju1uhO6Db+W/J6Fe535md2Ps6Zvyh3raGNbSEYPUWnelY7v\nX2Sr1KEWFFAh8HwIbWt8Zj84r/lJzAmAtTJEA2KGmowfDcU6ZJ77O0nEHXGLOqoH\nDDO0dRXbnkaY/7SEdcS7H3UsbgWSKV9vDeoWtxYPprE0yD4TykYNX4MUr9Qwzw6S\nBRFOeSuGXKGqJJGtnkPpRVeB7fbQ472O7b2aased9zffrpr6rDSmMgCdT7bjWsuh\nS7PeYH7YTfuV9B3b/kaioWe72sZE5x05zP6flBCOxY/7bbK2T/ZjgYHXvT3BX6+r-----END CERTIFICATE-----");

    public FileSystemCertificateRepository() throws GeneralSecurityException, IOException {
        this._certMap = this.loadCertificatesFromFile();
        this._certMap.put(this._ctasCert, new CertificateDefinition(this._ctasCert, true));
        this._certMap.put(this._patervaCaCert, new CertificateDefinition(this._patervaCaCert, true));
        this._certMap.put(this._ctasCert2019, new CertificateDefinition(this._ctasCert2019, true));
        this._certMap.put(this._patervaCaCert2019, new CertificateDefinition(this._patervaCaCert2019, true));
        if (Version.getCurrent().isDevBuild()) {
            FileSystemCertificateRepository.displayCertificateDevConsoleMessage(this._ctasCert, "Built-in CTAS certificate");
            FileSystemCertificateRepository.displayCertificateDevConsoleMessage(this._patervaCaCert, "Built-in Paterva CA certificate");
            FileSystemCertificateRepository.displayCertificateDevConsoleMessage(this._ctasCert2019, "Built-in CTAS 2019 certificate");
            FileSystemCertificateRepository.displayCertificateDevConsoleMessage(this._patervaCaCert2019, "Built-in Paterva CA 2019 certificate");
        }
    }

    private static void displayCertificateDevConsoleMessage(X509Certificate cert, String devConsoleMessage) {
        Date expiryDate = cert.getNotAfter();
        Date now = new Date();
        Calendar threeMonthFomNow = Calendar.getInstance();
        threeMonthFomNow.setTime(now);
        threeMonthFomNow.add(2, 3);
        if (!CertificateUtils.isValid(cert)) {
            int red = 31;
            LOG.log(Level.SEVERE, "\u001b[" + red + ";1m" + devConsoleMessage + " invalid\u001b[0m");
        } else if (threeMonthFomNow.getTime().before(expiryDate)) {
            int green = 32;
            LOG.log(Level.INFO, "\u001b[" + green + ";1m" + devConsoleMessage + " valid\u001b[0m");
        } else if (now.before(expiryDate)) {
            int yellow = 33;
            LOG.log(Level.WARNING, "\u001b[" + yellow + ";1m" + devConsoleMessage + " expires in less than 3 months\u001b[0m");
        } else {
            int red = 31;
            LOG.log(Level.SEVERE, "\u001b[" + red + ";1m" + devConsoleMessage + " has already expired\u001b[0m");
        }
    }

    private FileObject getOrCreateCertificateFolder() throws IOException {
        return FileUtilities.getOrCreate((FileObject)FileUtil.getConfigRoot(), (String)CERTIFICATES_FOLDER);
    }

    @Override
    public synchronized KeyStore getKeyStore(boolean trusted) throws GeneralSecurityException, IOException {
        X509Certificate cert;
        KeyStore keyStore = KeyStore.getInstance(KeyStore.getDefaultType());
        keyStore.load(null);
        if (!this._certMap.isEmpty()) {
            for (Map.Entry<X509Certificate, CertificateDefinition> entry : this._certMap.entrySet()) {
                cert = entry.getKey();
                CertificateDefinition certDef = entry.getValue();
                if (certDef.isTrusted() != trusted) continue;
                this.addToKeyStore(keyStore, cert);
            }
        } else if (trusted) {
            this.addToKeyStore(keyStore, this._ctasCert);
            this.addToKeyStore(keyStore, this._patervaCaCert);
            this.addToKeyStore(keyStore, this._ctasCert2019);
            this.addToKeyStore(keyStore, this._patervaCaCert2019);
        }
        if (LOG.isLoggable(Level.FINE)) {
            Enumeration<String> aliases = keyStore.aliases();
            while (aliases.hasMoreElements()) {
                String alias = aliases.nextElement();
                cert = (X509Certificate)keyStore.getCertificate(alias);
                LOG.log(Level.FINE, "{0} -> {1}", new Object[]{alias, CertificateUtils.toStringLine(cert)});
            }
        }
        return keyStore;
    }

    @Override
    public synchronized boolean contains(X509Certificate cert) {
        return this.contains(cert, true) || this.contains(cert, false);
    }

    @Override
    public synchronized boolean contains(X509Certificate cert, boolean trusted) {
        CertificateDefinition certDef = this._certMap.get(cert);
        boolean found = certDef != null && certDef.isTrusted() == trusted;
        return found;
    }

    @Override
    public synchronized boolean isCaValid(X509Certificate cert) {
        boolean valid = false;
        boolean isSigningCa = false;
        for (Map.Entry<X509Certificate, CertificateDefinition> entry : this._certMap.entrySet()) {
            isSigningCa = false;
            X509Certificate certInManager = entry.getKey();
            try {
                cert.verify(certInManager.getPublicKey());
                isSigningCa = true;
            }
            catch (InvalidKeyException | NoSuchAlgorithmException | NoSuchProviderException | SignatureException | CertificateException generalSecurityException) {
                // empty catch block
            }
            if (!isSigningCa) continue;
            valid = CertificateUtils.isValid(certInManager);
            break;
        }
        return valid;
    }

    @Override
    public synchronized boolean containsCa(X509Certificate cert) {
        return this.containsCa(cert, true) || this.containsCa(cert, false);
    }

    @Override
    public synchronized boolean containsCa(X509Certificate cert, boolean trusted) {
        boolean found = false;
        boolean isSigningCa = false;
        for (Map.Entry<X509Certificate, CertificateDefinition> entry : this._certMap.entrySet()) {
            isSigningCa = false;
            X509Certificate certInManager = entry.getKey();
            try {
                cert.verify(certInManager.getPublicKey());
                isSigningCa = true;
            }
            catch (InvalidKeyException | NoSuchAlgorithmException | NoSuchProviderException | SignatureException | CertificateException generalSecurityException) {
                // empty catch block
            }
            if (!isSigningCa) continue;
            CertificateDefinition certDefInManager = entry.getValue();
            found = certDefInManager != null && certDefInManager.isTrusted() == trusted;
            break;
        }
        return found;
    }

    @Override
    public synchronized CertificateDefinition get(X509Certificate cert) {
        CertificateDefinition certDef = this._certMap.get(cert);
        return certDef;
    }

    @Override
    public synchronized Collection<? extends X509Certificate> getAllDisplay() {
        ArrayList<X509Certificate> certificates = new ArrayList<X509Certificate>();
        certificates.addAll(this._certMap.keySet());
        certificates.remove(this._ctasCert);
        certificates.remove(this._patervaCaCert);
        certificates.remove(this._ctasCert2019);
        certificates.remove(this._patervaCaCert2019);
        return certificates;
    }

    private void updateMemory(Map<X509Certificate, CertificateDefinition> certMap, X509Certificate cert, boolean trusted, String url) throws GeneralSecurityException, IOException {
        CertificateDefinition certDef;
        String hostname = url = url == null ? null : CertificateUtils.stripUrl(url);
        if (url != null) {
            try {
                URL urlObject = new URL(url);
                hostname = urlObject.getHost();
            }
            catch (MalformedURLException ex) {
                LOG.log(Level.FINE, "Failed to get hostname for URL {0}: {2}", new Object[]{url, ex.toString()});
            }
        }
        if ((certDef = certMap.get(cert)) == null) {
            certDef = new CertificateDefinition(cert, trusted, hostname);
        } else {
            certDef.setTrusted(trusted);
            certDef.addHostname(hostname);
        }
        certMap.put(cert, certDef);
    }

    private void addToKeyStore(KeyStore keyStore, X509Certificate cert) throws GeneralSecurityException, IOException {
        int num = 0;
        String alias = this.createAlias(num);
        while (!this.isUniqueAlias(alias, keyStore)) {
            alias = this.createAlias(++num);
        }
        keyStore.setCertificateEntry(alias, cert);
    }

    private String createAlias(int num) {
        return String.format("Cert%04d", num);
    }

    private boolean isUniqueAlias(String alias, KeyStore keyStore) throws KeyStoreException {
        boolean uniqueAlias = false;
        Certificate checkCert = keyStore.getCertificate(alias);
        if (checkCert == null) {
            uniqueAlias = true;
        }
        return uniqueAlias;
    }

    @Override
    public synchronized void update(X509Certificate cert, boolean trusted, String url) {
        try {
            LOG.log(Level.FINE, "Update {0}: {2}", new Object[]{trusted ? ATTR_TRUSTED : "untrusted", CertificateUtils.toStringLine(cert)});
            this.updateMemory(this._certMap, cert, trusted, url);
            this.saveToFile(cert);
            this.fireItemChanged(cert);
        }
        catch (IOException | GeneralSecurityException ex) {
            Exceptions.printStackTrace((Throwable)ex);
        }
    }

    @Override
    public synchronized boolean remove(X509Certificate cert) {
        FileObject file;
        boolean isDeleted = false;
        CertificateDefinition prevCertDef = this._certMap.get(cert);
        if (prevCertDef != null && (file = prevCertDef.getFileObject()) != null) {
            try {
                file.delete();
                this._certMap.remove(cert);
                isDeleted = true;
                this.fireItemRemoved(cert);
            }
            catch (IOException ex) {
                Exceptions.printStackTrace((Throwable)ex);
            }
        }
        return isDeleted;
    }

    private Map<X509Certificate, CertificateDefinition> loadCertificatesFromFile() throws IOException {
        this._certMap = new HashMap<X509Certificate, CertificateDefinition>();
        FileObject folder = this.getOrCreateCertificateFolder();
        if (folder != null) {
            Enumeration fileEnum = folder.getData(false);
            while (fileEnum.hasMoreElements()) {
                FileObject file = (FileObject)fileEnum.nextElement();
                if (!EXTENSION.equals(file.getExt())) continue;
                CertificateDefinition certDef = this.loadFromFile(file);
                this._certMap.put(certDef.getCertificate(), certDef);
            }
        }
        return this._certMap;
    }

    private CertificateDefinition loadFromFile(FileObject file) throws IOException {
        Boolean trusted = (Boolean)file.getAttribute(ATTR_TRUSTED);
        X509Certificate cert = (X509Certificate)file.getAttribute(ATTR_X509CERTIFICATE);
        Object hostnamesObject = file.getAttribute(ATTR_HOSTNAMES);
        List hostnames = hostnamesObject == null ? null : (List)hostnamesObject;
        CertificateDefinition certDef = new CertificateDefinition(cert, trusted, file, hostnames);
        return certDef;
    }

    private void saveToFile(X509Certificate cert) throws GeneralSecurityException, IOException {
        FileObject folder = this.getOrCreateCertificateFolder();
        CertificateDefinition certDef = this._certMap.get(cert);
        FileSystemCertificateRepository.saveToFile(folder, certDef);
    }

    private static void saveToFile(FileObject folder, CertificateDefinition certDef) throws IOException {
        String name;
        if (folder != null && certDef != null && !StringUtilities.isNullOrEmpty((String)(name = CertificateUtils.getId(certDef)))) {
            String fileName = String.format("%s.%s", name, EXTENSION);
            FileSystemCertificateRepository.deleteFile(folder, name, fileName);
            try {
                String suffix = DigestUtils.sha1Hex((byte[])certDef.getCertificate().getEncoded());
                name = name + "_" + suffix.substring(0, Math.min(10, suffix.length() - 1));
            }
            catch (Exception suffix) {
                // empty catch block
            }
            fileName = String.format("%s.%s", name, EXTENSION);
            FileSystemCertificateRepository.deleteFile(folder, name, fileName);
            FileObject file = folder.createData(fileName);
            certDef.setFileObject(file);
            file.setAttribute(ATTR_TRUSTED, (Object)certDef.isTrusted());
            file.setAttribute(ATTR_X509CERTIFICATE, (Object)certDef.getCertificate());
            List<String> hostnames = certDef.getHostnames();
            file.setAttribute(ATTR_HOSTNAMES, hostnames);
        }
    }

    private static void deleteFile(FileObject folder, String name, String fileName) throws IOException {
        File fileCaseInsensitive;
        FileObject file = folder.getFileObject(fileName);
        if (file != null) {
            file.delete();
        }
        if ((fileCaseInsensitive = FileUtilities.getFileFromFileObject((FileObject)folder, (String)name, (String)EXTENSION)).exists()) {
            fileCaseInsensitive.delete();
        }
    }

    private X509Certificate toCertificate(String certStr) throws CertificateException {
        ByteArrayInputStream derInputStream = new ByteArrayInputStream(certStr.getBytes());
        CertificateFactory certificateFactory = CertificateFactory.getInstance("X.509");
        X509Certificate cert = (X509Certificate)certificateFactory.generateCertificate(derInputStream);
        return cert;
    }
}

